/*
 * @(#)OMAssociationFigure.java  1.0  2. Februar 2004
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */
package ch.hslu.cm.oo.diagram;

import ch.hslu.cm.simulation.*;
import ch.hslu.cm.*;
import ch.hslu.cm.oo.*;
import java.io.IOException;
import org.jhotdraw.app.action.ActionUtil;
import org.jhotdraw.draw.*;
import ch.hslu.cm.oo.objectmodel.*;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.beans.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import java.awt.geom.*;
import org.jhotdraw.draw.decoration.ArrowTip;
import org.jhotdraw.draw.decoration.GeneralPathTip;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.layouter.LocatorLayouter;
import org.jhotdraw.draw.locator.BezierLabelLocator;
import static org.jhotdraw.draw.AttributeKeys.*;

/**
 * OMAssociationFigure visually represents a {@link OMAssociation}.
 * 
 * 
 * @author Werner Randelshofer
 * @version 1.0 2. Februar 2004  Created.
 */
public class OMAssociationFigure extends LabeledLineConnectionFigure
        implements DiagramFigure, PropertyChangeListener {

    private OMAssociation model;
    private static GeneralPathTip crossTip;
    private static ArrowTip arrowTip = new ArrowTip(Math.PI / 11, 13, 0, false, true, true);

    /**
     * This adapter is used, to connect a TextFigure for an attribute with
     * the SimulatedObject model.
     */
    private class LabelAdapter extends FigureAdapter {

        private boolean owner;

        public LabelAdapter(boolean owner) {
            this.owner = owner;
        }

        @Override
        public void attributeChanged(FigureEvent evt) {
            if (evt.getAttribute().equals(TEXT)) {
                if (owner) {
                    model.setOwnerLabel((String) evt.getNewValue());
                } else {
                    model.setMemberLabel((String) evt.getNewValue());
                }
            }
        }
    }

    static {
        Path2D.Double cross = new Path2D.Double();
        cross.moveTo(-4, 8);
        cross.lineTo(4, 16);
        cross.moveTo(4, 8);
        cross.lineTo(-4, 16);
        crossTip = new GeneralPathTip(cross, 0, false, true, true);
    }

    /** Creates a new instance. */
    public OMAssociationFigure() {
        setLayouter(new LocatorLayouter());
        setModel(new OMAssociation());

        setAttributeEnabled(END_DECORATION, false);
        setAttributeEnabled(START_DECORATION, false);
        setAttributeEnabled(STROKE_DASHES, false);
        setAttributeEnabled(FONT_BOLD, false);
        setAttributeEnabled(FONT_ITALIC, false);
        setAttributeEnabled(FONT_UNDERLINE, false);
    }

    @Override
    public OMAssociation getModel() {
        return model;
    }

    public void setModel(OMAssociation m) {
        if (model != null) {
            model.removePropertyChangeListener(this);
        }
        model = m;
        if (model != null) {
            model.addPropertyChangeListener(this);
        }
        updateLabels();
    }

    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }

    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }

    private void updateLabels() {
        willChange();
        basicRemoveAllChildren();

        setAttributeEnabled(START_DECORATION, true);
        setAttributeEnabled(END_DECORATION, true);

        if (model == null) {
            START_DECORATION.set(this, null);
            END_DECORATION.set(this, null);
        } else {
            if (model.getOwnerMultiplicity() != 0) {
                TextFigure ownerMultiplicity = new TextFigure();
                ownerMultiplicity.set(LocatorLayouter.LAYOUT_LOCATOR, new BezierLabelLocator(0, -Math.PI / 4, 8));
                ownerMultiplicity.setEditable(false);
                switch (model.getOwnerMultiplicity()) {
                    case -1:
                        ownerMultiplicity.setText("*");
                        break;
                    default:
                        ownerMultiplicity.setText(Integer.toString(model.getOwnerMultiplicity()));
                        break;
                }
                basicAdd(ownerMultiplicity);
            }
            if (model.getOwnerMultiplicity() != 0) {
                TextFigure memberMultiplicity = new TextFigure();
                memberMultiplicity.set(LocatorLayouter.LAYOUT_LOCATOR, new BezierLabelLocator(1, Math.PI + Math.PI / 4, 8));
                memberMultiplicity.setEditable(false);
                switch (model.getMemberMultiplicity()) {
                    case -1:
                        memberMultiplicity.setText("*");
                        break;
                    default:
                        memberMultiplicity.setText(Integer.toString(model.getMemberMultiplicity()));
                        break;
                }
                basicAdd(memberMultiplicity);
            }



            if (model.isOwnerTraversable()) {
                START_DECORATION.set(this, arrowTip);
                TextFigure ownerLabel = new TextFigure(model.getOwnerLabel());
                ownerLabel.set(LocatorLayouter.LAYOUT_LOCATOR, new BezierLabelLocator(0, Math.PI / 4, 8));
                ownerLabel.addFigureListener(new LabelAdapter(true));
                basicAdd(ownerLabel);
            } else {
                // START_DECORATION.set(this, crossTip);
                START_DECORATION.set(this, null);
            }

            if (model.isMemberTraversable()) {
                END_DECORATION.set(this, arrowTip);
                TextFigure memberLabel = new TextFigure(model.getMemberLabel());
                memberLabel.set(LocatorLayouter.LAYOUT_LOCATOR, new BezierLabelLocator(1, Math.PI - Math.PI / 4, 8));
                memberLabel.addFigureListener(new LabelAdapter(false));
                basicAdd(memberLabel);
            } else {
                // END_DECORATION.set(this, crossTip);
                END_DECORATION.set(this, null);
            }
        }

        setAttributeEnabled(START_DECORATION, false);
        setAttributeEnabled(END_DECORATION, false);

        layout();
        changed();
    }

    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;

        Figure item;

        action = new AbstractAction("1:1") {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setOwnerMultiplicity(1);
                model.setMemberMultiplicity(1);
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, "Multiplizit\u00e4t");
        actions.add(action);

        action = new AbstractAction("1:*") {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setOwnerMultiplicity(1);
                model.setMemberMultiplicity(-1);
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, "Multiplizit\u00e4t");
        actions.add(action);

        action = new AbstractAction("*:1") {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setOwnerMultiplicity(-1);
                model.setMemberMultiplicity(1);
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, "Multiplizit\u00e4t");
        actions.add(action);

        action = new AbstractAction("*:*") {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setOwnerMultiplicity(-1);
                model.setMemberMultiplicity(-1);
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, "Multiplizit\u00e4t");
        actions.add(action);

        action = new AbstractAction(model.isMemberTraversable() ? "Mache Ende nicht traversierbar" : "Mache Ende traversierbar") {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setMemberTraversable(!model.isMemberTraversable());
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, "Traversierbarkeit");
        actions.add(action);

        action = new AbstractAction(model.isOwnerTraversable() ? "Mache Start nicht traversierbar" : "Mache Start traversierbar") {

            @Override
            public void actionPerformed(ActionEvent event) {
                model.setOwnerTraversable(!model.isOwnerTraversable());
            }
        };
        action.putValue(ActionUtil.SUBMENU_KEY, "Traversierbarkeit");
        actions.add(action);

        return actions;
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        updateLabels();
    }

    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }

    private Simulation getSimulation() {
        return (getDiagram() == null) ? null : getDiagram().getSimulation();
    }

    public SimulatedRelationship getSimulatedObject() {
        return model;
    }

    @Override
    public OMAssociationFigure clone() {
        OMAssociationFigure that = (OMAssociationFigure) super.clone();
        that.setModel((OMAssociation) this.getModel().clone());
        return that;
    }

    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    protected void handleDisconnect(Figure start, Figure end) {
        model.setEnd(null);
        model.setStart(null);
    }

    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    protected void handleConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure)
                && (end instanceof DiagramFigure)) {
            DiagramFigure sf = (DiagramFigure) start;
            DiagramFigure ef = (DiagramFigure) end;

            model.setEnd((SimulatedElement) ef.getModel());
            model.setStart((SimulatedElement) sf.getModel());
        }
    }

    /**
     * Checks if two figures can be connected. Implement this method
     * to constrain the allowed connections between figures.
     */
    public boolean canConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure)
                && (end instanceof DiagramFigure)) {

            DiagramFigure sf = (DiagramFigure) start;
            DiagramFigure ef = (DiagramFigure) end;

            return model.canConnect((SimulatedElement) sf.getModel(), (SimulatedElement) ef.getModel());
        }

        return false;
    }

    public boolean canConnect(Figure start) {
        if (start instanceof DiagramFigure) {
            DiagramFigure sf = (DiagramFigure) start;
            if (sf.getModel() instanceof SimulatedElement) {
                return model.canConnect((SimulatedElement) sf.getModel());
            }
        }
        return false;
    }

    public int getConnectionCount() {
        return 0;
    }

    public int getConnectionIndex(DiagramFigure f) {
        return 0;
    }

    @Override
    public void read(DOMInput in) throws IOException {
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((OMAssociation) in.readObject(0));
        in.closeElement();
        readPoints(in);
        readAttributes(in);
    }

    @Override
    public void write(DOMOutput out) throws IOException {
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
        writePoints(out);
        writeAttributes(out);
    }

    @Override
    public int getLayer() {
        return ClassDiagram.LINK_LAYER;
    }
}
